/*
 * FinalProjectFairComRead.c
 *
 *  Created on: May 9, 2015
 *      Author: jcmx9
 */

#ifdef _WIN32_WCE
#undef UNICODE
#undef _UNICODE
#define main my_main
#endif


/* Preprocessor definitions and includes */

#include "ctdbsdk.h" 		// c-tree headers
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <sys/types.h>
#include <sys/stat.h>		//for open()/write() mode
#include <fcntl.h>			//for open()/write()
#include <err.h>			//for open() error test
#include <sys/time.h>		//for time() function
#include <unistd.h>
#include <sys/select.h>		//for select() function

#define END_OF_FILE INOT_ERR  /* INOT_ERR is ctree's 101 error. See cterrc.h */


/* Global declarations */

CTHANDLE hSession;
CTHANDLE hDatabase;
CTHANDLE hTable;
CTHANDLE hRecord;


/* Function declarations */

#ifdef PROTOTYPE
VOID Initialize(VOID), Define(VOID), Manage(VOID), Done(VOID);
VOID Display_Records(VOID), ReadDistanceChange (VOID);
VOID Delete_Records(CTHANDLE), Check_Table_Mode(CTHANDLE);
VOID Handle_Error(CTSTRING);
VOID error (const char *msg);
#else
VOID Initialize(), Define(), Manage(), Done();
VOID Add_Records(), Display_Records(), Add_Timestamps_Record (), ReadDistanceChange ();
VOID Delete_Records(), Check_Table_Mode();
VOID Handle_Error();
VOID error (msg);
#endif


/*
 * main()
 *
 * Call functions in steps
 */

#ifdef PROTOTYPE
NINT main (NINT argc, pTEXT argv[])
#else
NINT main (argc, argv)
NINT argc;
TEXT argv[];
#endif
{
   Initialize();

   Define();

   Manage();

   Done();

   printf("\nPress <ENTER> key to exit . . .\n");
#ifndef ctPortWINCE
   getchar();
#endif

   return(0);
}


/*
 * Initialize()
 *
 * Perform the minimum requirement of logging onto the c-tree Server
 */

#ifdef PROTOTYPE
VOID Initialize(VOID)
#else
VOID Initialize()
#endif
{
   CTDBRET  retval;

   printf("INIT\n");

   if ((retval = ctdbStartDatabaseEngine())) 				//using the Server DLL model to start the underlying Server.
	Handle_Error("Initialize(): ctdbStartDatabaseEngine()");

   /* allocate session handle */
   if ((hSession = ctdbAllocSession(CTSESSION_CTREE)) == NULL)			//allocate session
      Handle_Error("Initialize(): ctdbAllocSession()");

   hDatabase = hSession; /* database not used in this program */

   /* connect to server */
   printf("\tLogon to server...\n");
   if (ctdbLogon(hSession, "FAIRCOMS", "ADMIN", "ADMIN"))				//login database by default id and passcode
      Handle_Error("Initialize(): ctdbLogon()");
}


/*
 * Define()
 *
 * Open the table, if it exists. Otherwise create and open the table
 */

#ifdef PROTOTYPE
VOID Define(VOID)
#else
VOID Define()
#endif
{
	CTDBRET  retval;
	CTHANDLE hField1, hField2;

   printf("DEFINE\n");

   /* allocate a table handle */
   if ((hTable = ctdbAllocTable(hDatabase)) == NULL)
	   Handle_Error("Define(); ctdbAllocTable()");

   /* open table */
   printf("\tOpen table...\n");
   if (ctdbOpenTable(hTable, "TimeStamp", CTOPEN_NORMAL))			//Table Name: TimeStamp
   {
      /* define table fields, two fields,  */
      printf("\tAdd fields...\n");
      hField1 = ctdbAddField(hTable, "Date", CT_STRING, 15);		//CT_STRING, char* type to store date string
      hField2 = ctdbAddField(hTable, "Time", CT_STRING, 15);		//CT_STRING, char* type to store time string

      if (!hField1 || !hField2)										//error check
         Handle_Error("Define(); ctdbAddField()");

      /* create table */
      printf("\tCreate table...\n");
      if (ctdbCreateTable(hTable, "TimeStamp", CTOPEN_NORMAL))		//if the table does not exist, create one
         Handle_Error("Define(); ctdbCreateTable()");

      if (ctdbOpenTable(hTable, "TimeStamp", CTOPEN_NORMAL))
         Handle_Error("Define(); ctdbOpenTable()");
   }
   else
   {
      Check_Table_Mode(hTable);
   }

}

/*
 * Manage()
 *
 * This function performs simple record functions of add, delete and gets
 */

#ifdef PROTOTYPE
VOID Manage(VOID)
#else
VOID Manage()
#endif
{
   printf("MANAGE\n");

   /* allocate a record handle */
   if ((hRecord = ctdbAllocRecord(hTable)) == NULL)			//allocate a record handler to operate record
      Handle_Error("Manage(): ctdbAllocRecord()");

   /* delete any existing records */
   Delete_Records(hRecord);									//clean records before store

   ReadDistanceChange ();									//read distance from kernel by fifo and compare

   /* display contents of table */
   Display_Records();										//display records after record finish
}


/*
 * Done()
 *
 * This function handles the housekeeping of closing tables and
 * freeing of associated memory
 */

#ifdef PROTOTYPE
VOID Done(VOID)
#else
VOID Done()
#endif
{
   printf("DONE\n");

   /* close table */
   printf("\tClose table...\n");
   if (ctdbCloseTable(hTable))							//close table first
      Handle_Error("Done(): ctdbCloseTable()");

   /* logout */
   printf("\tLogout...\n");
   if (ctdbLogout(hSession))							//logout the session
      Handle_Error("Done(): ctdbLogout()");				//no "database" is created or open in this case so don't need close

   /* free handles */
   ctdbFreeRecord(hRecord);								//free record handler
   ctdbFreeTable(hTable);								//free table handler
   ctdbFreeDatabase(hDatabase);							//free database handler
   ctdbFreeSession(hSession);							//free session handler

   //if we are linked to the Server DLL, then we should stop our Server at the end of the program.
   ctdbStopDatabaseEngine();			//stop database engine because we are linked to the Server DLL before
}


/*
 * Check_Table_Mode()
 *
 * Check if existing table has transaction processing flag enabled.
 * If a table is under transaction processing control, ctdbGetError()modify the
 * table mode to disable transaction processing
 */

#ifdef PROTOTYPE
VOID Check_Table_Mode(CTHANDLE hTable)
#else
VOID Check_Table_Mode(hTable)
CTHANDLE hTable;
#endif
{
   CTCREATE_MODE mode;

   /* get table create mode */
   mode = ctdbGetTableCreateMode(hTable);

   /* check if table is under transaction processing control */
   if ((mode & CTCREATE_TRNLOG))
   {
      /* change file mode to disable transaction processing */
      mode ^= CTCREATE_TRNLOG;
      if (ctdbUpdateCreateMode(hTable, mode) != CTDBRET_OK)
         Handle_Error("Check_Table_Mode(); ctdbUpdateCreateMode");
   }
}


/*
 * Delete_Records()
 *
 * This function deletes all the records in the table
 */

#ifdef PROTOTYPE
VOID Delete_Records(CTHANDLE hRecord)
#else
VOID Delete_Records(hRecord)
CTHANDLE hRecord;
#endif
{
   CTDBRET  retval;							//retval of function, for error checking purpose
   CTBOOL   empty;

   printf("\tDelete records...\n");

   empty = NO;
   retval = ctdbFirstRecord(hRecord);		//point the record handler to the first record
   if (retval != CTDBRET_OK)
   {
      if (retval == END_OF_FILE)
         empty = YES;						//no record is in the table
      else
         Handle_Error("Delete_Records(): ctdbFirstRecord()");
   }

   while (empty == NO) /* while table is not empty */
   {
      /* delete record */
      if (ctdbDeleteRecord(hRecord))
         Handle_Error("Delete_Records(): ctdbDeleteRecord()");

      /* read next record */
      retval = ctdbNextRecord(hRecord);
      if (retval != CTDBRET_OK)
      {
         if (retval == END_OF_FILE)
            empty = YES;
         else
            Handle_Error("Delete_Records(): ctdbNextRecord()");			//check error every time
      }
   }
}


/*
 * ReadDistanceChange ()
 *
 * For all testing purposes
 */

#ifdef PROTOTYPE
VOID ReadDistanceChange (VOID)
#else
VOID ReadDistanceChange ()
#endif
{
	//faircom variable
	CTDBRET  retval;
	CTDATE nowDate;
	CTTIME nowTime;
	TEXT nowDateString[15+1];		//string to
	TEXT nowTimeString[15+1];

	double prevDist = 0;			//initiate previous distance as 0 for comparing convenience
	double newDist;					//variable to store incoming distance
	int edgeFlag = 2;				//invalid number for if statement
	bool startFlag = false;			//double protect from receiving out of order timestamps
	bool endFlag = false;			//double protect from receiving out of order timestamps
	//ts_start to store start timestamp, ts_end to store end timestamp, time_discard to store out of order timestamp
	struct timeval ts_start, ts_end, time_discard;
	double duration;				//variable to store the duration of rising edge and falling edge of "Echo" pin
	int fifo0 = open("/dev/rtf/0", O_RDWR);				//fifo0 to receive edge flag from kernel
	int fifo1 = open("/dev/rtf/1", O_RDWR);				//fifo1 to receive timestamps kernel

	//for timeout option, timeout is set for demonstration purpose
	fd_set set;
	struct timespec timeout;
	int rv;

	FD_ZERO(&set); 				//clear the set
 	FD_SET(fifo1, &set);		//add file descriptor, the fifo1 to the set,
 	timeout.tv_sec = 5;			//5 seconds timeout for reading fifo1
	timeout.tv_nsec = 0;

	while (1){
		startFlag = false;		//set the flag back to false at the beginning of
		endFlag = false;
		//set timeout for fifo1
		rv = pselect(fifo1 + 1, &set, NULL, NULL, &timeout, NULL);
		if (rv == -1)
		    err(1, "ReadDistanceChange(): pselect()"); 			//error occur in pselect()
		else if (rv == 0){
		    printf("timeout...\n"); 			//when timeout, return the read distance function
			return;
		}
		else{
			//if no error and not timeout yet, read the edge flag fifo1
			if (read (fifo1, &edgeFlag, sizeof(int)) != sizeof (int))
				err(1, "ReadDistanceChange (): read() fifo1");
		}

		if (edgeFlag == 0){							//error, should start from rising edge
			//read fifo0 and discard this timestamp because it start with falling edge
			if (read (fifo0, &time_discard, sizeof(time_discard)) != sizeof (time_discard))
				err(1, "ReadDistanceChange (): read() fifo0");
			startFlag = false;
			endFlag = false;
		}
		else if (edgeFlag == 1){					//correct, start recording from rising edge
			//get timestamp of rising edge
			if (read (fifo0, &ts_start, sizeof(ts_start)) != sizeof (ts_start))
				err(1, "ReadDistanceChange (): read() fifo0");
			startFlag = true;

			//read flag fifo 1 to make sure next timestamps is for falling edge
			if (read (fifo1, &edgeFlag, sizeof(int)) != sizeof (int))
				err(1, "ReadDistanceChange (): read() fifo1");

			if (edgeFlag == 1){				//error, cannot be two continuously rising edge
				if (read (fifo0, &time_discard, sizeof(time_discard)) != sizeof (time_discard))
					err(1, "ReadDistanceChange (): read() fifo0");
				startFlag = false;
				endFlag = false;
			}
			else if (edgeFlag == 0){		//correct, record timestamp for falling edge
				if (read (fifo0, &ts_end, sizeof(ts_end)) != sizeof (ts_end))
					err(1, "ReadDistanceChange (): read() fifo0");
				endFlag = true;
			}
		}


		if (startFlag & endFlag){
		//calculate in integer to prevent decimal number (microsecond) lost in rounding up, for accuracy
		duration = abs(ts_end.tv_sec * 1000000 + ts_end.tv_usec - ts_start.tv_sec * 1000000 - ts_start.tv_usec) / 1000;		//in ms
		newDist = duration / 1000 * 34000 / 2;		//speed of sound: 34000cm/millisecond
		}

		if (abs(newDist - prevDist) > 20000){		//20000cm for threshold because I am using push button to similate the "Echo" pulse
			printf ("\nDistance Changed @\n");
			retval = 0;
			retval |= ctdbCurrentDate (&nowDate);	//ctdb function to get local date
			retval |= ctdbCurrentTime (&nowTime);	//ctdb function to get local time
			if (retval != CTDBRET_OK){
					Handle_Error("ReadDistanceChange (): ctdbCurrentDate (), ctdbCurrentTime ()");
			}

			//parse date to string in MM/DD/CCYY format
			retval = ctdbDateToString (nowDate, CTDATE_MDCY, nowDateString, sizeof (nowDateString));
			//parse time to string in HH:MM:SS AP format
			retval = ctdbTimeToString (nowTime, CTTIME_HMSP, nowTimeString, sizeof (nowTimeString));

			printf ("Date: %s.\n", nowDateString);
			printf ("Time: %s.\n\n", nowTimeString);

			if (retval != CTDBRET_OK){
				Handle_Error("ReadDistanceChange (): ctdbDateToString(), ctdbTimeToStrin() ");
			}

			/* clear record buffer */
			ctdbClearRecord(hRecord);

			//populate record buffer with data
			retval = 0;
			//set the new inserted record field 0 as String to match the table
			retval |= ctdbSetFieldAsString(hRecord, 0, nowDateString);
			//set the new inserted record field 1 as String to match the table
			retval |= ctdbSetFieldAsString(hRecord, 1, nowTimeString);

			if (retval)
				Handle_Error("ReadDistanceChange(): ctdbSetFieldAsString()");

			/* add record */
			if (ctdbWriteRecord(hRecord))
			   Handle_Error("ReadDistanceChange(): ctdbWriteRecord()");
		}
		else
			printf ("Distance changes within threshold.\n");

		prevDist = newDist;						//save the incoming distance as previous distance for next loop

	}
}


/*
 * Display_Records()
 *
 * This function displays the contents of a table. ctdbFirstRecord() and
 * ctdbNextRecord() fetch the record. Then each field is parsed and displayed
 */

#ifdef PROTOTYPE
VOID Display_Records(VOID)
#else
VOID Display_Records()
#endif
{
   CTDBRET  retval;
   TEXT     nowdate[15+1];
   TEXT     nowtime[15+1];

   printf("\tDisplay records...");

   /* read first record */
   retval = ctdbFirstRecord(hRecord);					//find the first record in the table
   if (retval != CTDBRET_OK)
      Handle_Error("Display_Records(): ctdbFirstRecord()");			//if the table is empty

   while (retval != END_OF_FILE)
   {
      retval = 0;
      //retrieve the data in table as string
      retval |= ctdbGetFieldAsString(hRecord, 0, nowdate, sizeof (nowdate));
      retval |= ctdbGetFieldAsString(hRecord, 1, nowtime, sizeof (nowtime));
      if (retval)
         Handle_Error("Display_Records(): ctdbGetFieldAsString()");

      printf("\n\t\t%s%s\n", nowdate, nowtime);

      /* read next record */
      retval = ctdbNextRecord(hRecord);					//point the record handler to the next record
      if (retval == END_OF_FILE)
         break;   /* reached end of file */

      if (retval != CTDBRET_OK)
         Handle_Error("Display_Records(): ctdbNextRecord()");
   }
}


/*
 * Handle_Error()
 *
 * This function is a common bailout routine. It displays an error message
 * allowing the user to acknowledge before terminating the application
 */

#ifdef PROTOTYPE
VOID Handle_Error(CTSTRING errmsg)
#else
VOID Handle_Error(errmsg)
CTSTRING errmsg;
#endif
{
   printf("\nERROR: [%d] - %s \n", ctdbGetError(hSession), errmsg);
   printf("*** Execution aborted *** \nPress <ENTER> key to exit...");

   ctdbLogout(hSession);

   ctdbFreeRecord(hRecord);
   ctdbFreeTable(hTable);
   ctdbFreeSession(hSession);

   getchar();

   exit(1);
}

/*
 * error()
 *
 * This function handles error message in ReadDistanceChange();
 *
 */
#ifdef PROTOTYPE
VOID error (const char *msg)
#else
VOID error (msg)
#endif
{
    perror(msg);
    exit(0);
}

/* end of FinalProjectFairComRead.c */
